/*
   author:     Rony G. Flatscher
   name:       traceutils.cls
   purpose:    utilities for working with TraceObjects
   date:       2024-02-01 - 2024-02-03
   license:    AL 2.0, CPL 1.0
   status:     work in progress
   version:    0.1
*/

pkgLocal=.context~package~local  -- get access to this package local directory

pkgLocal~cr.lf="0d0a"x

   -- translate index names to title names
pkgLocal["ENTRY.NAMES"] = .stringTable~of(         -
            ("ATTRIBUTEPOOL"  ,"attributePool"  ), -
            ("HASOBJECTLOCK"  ,"hasObjectLock"  ), -
            ("INTERPRETER"    ,"interpreter"    ), -
            ("INVOCATION"     ,"invocation"     ), -
            ("ISGUARDED"      ,"isGuarded"      ), -
            ("NR"             ,"nr"             ), -
            ("OBJECTLOCKCOUNT","objectLockCount"), -
            ("OPTION"         ,"option"         ), -
            ("THREAD"         ,"thread"         ), -
            ("TIMESTAMP"      ,"timestamp"      ), -
            ("TRACELINE"      ,"traceline"      )  -
        )

pkgLocal["ENTRY.NAMES.ANNOTATED"] = .entry.names~copy   -
            ~~setEntry("HASGUARDSTATE","hasGuardState") -
            ~~setEntry("ISRUNNING"    ,"isRunning"    )

   -- order to use for creating JSON/CSV files fields/columns
pkgLocal["FIELD.ORDER"] = ("OPTION", "NR", "TIMESTAMP", "INTERPRETER",  -
                          "THREAD", "INVOCATION", "ISGUARDED",          -
                          "ATTRIBUTEPOOL", "OBJECTLOCKCOUNT",           -
                          "HASOBJECTLOCK", "TRACELINE")

pkgLocal["FIELD.ORDER.ANNOTATED"] = .field.order~copy              -
                                    ~~append("HASGUARDSTATE") -
                                    ~~append("ISRUNNING")

   -- entries that are only available if method routine gets traced (object-related)
pkgLocal["OBJECT.RELATED"] = .set~of("ISGUARDED", "ATTRIBUTEPOOL",      -
                                     "OBJECTLOCKCOUNT", "HASOBJECTLOCK")

pkgLocal["OBJECT.RELATED.ANNOTATED"] = .object.related~copy           -
                                       ~~put("HASGUARDSTATE") -
                                       ~~put("ISRUNNING")


::requires "json.cls"
::requires "csvStream.cls"

/** Right adjusts a value and returns it.
 *
 * @param value  the value to adjust
 * @param width  the width to use for adjustment, defaults to 3
 * @return "value" adjusted according to "width", or if longer than "width"
 *         "value" unchanged
 */
::routine adjRight            public
  use strict arg value, width=3
  if value~length>=width then return value
  return value~right(width)

/** Left adjusts a value and returns it.
 *
 * @param value  the value to adjust
 * @param width  the width to use for adjustment, defaults to 3
 * @return "value" adjusted according to "width", or if longer than "width"
 *         "value" unchanged
 */
   -- left adjust, if exceeding width return value unchanged
::routine adjLeft             public
  use strict arg value, width=3
  if value~length>=width then return value
  return value~left(width)


/** Create a JSON rendering from supplied collection.
 *
 * @param collection
 * @param option "M[inimal]" ... minimal, "C[RLF]" ... minimal, but CR-LF after each traceObject,
 *               "H[uman]" ... human legible
 *
 * @return a string encoded as JSON representing all of the traceObjects in "collection"
*/
::routine toJson              public
  use strict arg collection, option="C"
  opt=option~left(1)~upper
  if pos(opt,"MCH")=0 then
     raise syntax 93.914 array (1, '"M" (minimal), "C" (minimal with CRLF) or "H" (human legible)', arg(1))

  indent="   "    -- if human legible use CRLF and indentation
  indent2=indent~copies(2)
  mb=.MutableBuffer~new
  mb~append("[")
  ch?=pos(opt,"CH")>0

  tmpFieldOrder=.field.order
  traceObj=collection[1]      -- check whether traceObjects have annotation entries
  if \traceObj~isNil, traceObj~hasEntry("isRunning") then
     tmpFieldOrder=.field.order.annotated

  maxOrder=tmpFieldOrder~items   -- determine number of fields
  maxColl=collection~items       -- determine number of traceobjects

  if ch? then mb~append(.cr.lf)
  do counter c traceObj over collection
     if opt='H' then mb~append(indent)
     noObjInfos?=traceObj["ATTRIBUTEPOOL"]~isNil -- object related entries available?

     mb~append("{")
     if opt='H' then mb~append(.cr.lf,indent2)

     do counter c2 idx over tmpFieldOrder   -- show entries
        if noObjInfos?, .object.related~hasIndex(idx) -- skip?
           then iterate
        mb~append('"',.entry.names[idx],'":')
        if ch? then mb~append(' ')

        val=traceObj[idx]
        select case idx
        when "ATTRIBUTEPOOL", "INTERPRETER", "INVOCATION", "NR",  -
             "OBJECTLOCKCOUNT", "THREAD" then mb~append(val)

        when "HASOBJECTLOCK","ISGUARDED",          -
             "HASGUARDSTATE","ISRUNNING" then mb~append(val~?("true","false"))

        otherwise -- "TIMESPAN", "TRACELINE", "OPTION"
             mb~append('"',escJson(val),'"')
        end

        if c2<>maxOrder then
        do
           mb~append(",")
           if opt='H' then mb~append(.cr.lf,indent2)
        end
        -- if opt='H' then mb~append(.cr.lf,indent2)
     end
     if opt='H' then mb~append(.cr.lf,indent)
     mb~append("}")
     if maxColl<>c then     -- not last item?
        mb~append(",")

     if ch? then mb~append(.cr.lf)
  end
  mb~append("]")
  return mb~string

escJson: procedure      -- escape double quote and backslash in JSON style
  parse arg val
  return val~changeStr('\','\\') ~changeStr('"', '\"')


::routine toJsonFile          public
  use strict arg fn, arr, option='C'   -- Crlf: almost minimal, but each traceObject in its own line

  jsonData=toJson(arr,option)    -- turn traceObjects into json
  call writeFile fn, jsonData    -- write to file
  return


::routine fromJsonFile        public
  use strict arg fn
  inArray=.json~fromJsonFile(fn)
  toArray=.array~new
  do inObj over inArray
     traceObj=.traceObject~new
     do idx over inObj~allIndexes
        if idx~upper="TIMESTAMP" then
           traceObj~setEntry(idx,.dateTime~fromIsoDate(inObj[idx]))
        else
           traceObj~setEntry(idx,inObj[idx])
     end
     toArray~append(traceObj)
  end
  return toArray



/* ========================================================================= */

/** Create a CSV rendering from supplied collection.
 *
 * @param collection the collection of traceObjects
 * @param createTitle? indicates whether to supply a title line (default: .true)
 *
 * @return a string formatted as CSV (comma separated values) representing
 *         the traceObjects supplied via "collection"
 * @return a string encoded as CSV (comma separated values representing all of
 *         the traceObjects in "collection"
*/
::routine toCSV               public
  use strict arg collection, createTitle?=.true
  crlf  ="0d0a"x  -- CR-LF
  mb=.MutableBuffer~new

  tmpFieldOrder=.field.order
  traceObj=collection[1]      -- check whether traceObjects have annotation entries
  if \traceObj~isNil, traceObj~hasEntry("isRunning") then
     tmpFieldOrder=.field.order.annotated

  maxOrder=tmpFieldOrder~items    -- determine number of fields
  maxColl=collection~items       -- determine number of traceobjects
  if createTitle? then
     call createTitleLine mb, maxOrder, tmpFieldOrder

  do counter c traceObj over collection
     noObjInfos?=traceObj["ATTRIBUTEPOOL"]~isNil -- object related entries available?
     do counter c2 idx over tmpFieldOrder   -- show entries
        if noObjInfos?, .object.related~hasIndex(idx) then  -- skip?
        do
           mb~append(',')        -- empty column
           iterate
        end
        mb~append(quote(traceObj[idx],.true))   -- escape as traceline may contain quotes
        if c2<>maxOrder then mb~append(",")
     end
     if maxColl<>c then    -- not last traceObject ?
        mb~append(.cr.lf)
  end
  return mb~string

createTitleLine: procedure
   use arg mb, maxOrder, tmpFieldOrder
   do counter c idx over tmpFieldOrder   -- show entries
     mb~append(.entry.names[idx])
     if c<>maxOrder then mb~append(",")
   end
   return mb~append(.cr.lf)


::routine toCsvFile           public
  use strict arg fn, arr, option=.true -- .true (write header line, default)

  csvData=toCsv(arr,option)      -- turn traceObjects into json
  call writeFile fn, csvData     -- write to file

  return

-- see ooRexx documentation "Rexx Extension Library Reference" (rexxextensions.pdf)
::routine fromCsvFile        public
  use strict arg fn
  toArray=.array~new
  csv=.csvStream~new(fn)
  csv~skipHeaders=.false                  -- we want the header row as normal line
  csv~open('read')
  arrHeaderFields=csv~CSVLinein
  do counter c val over arrHeaderFields   -- uppercase field names
     arrHeaderFields[c]=val~upper
  end

  if arrHeaderFields[1]<>"OPTION" then    -- no header fields from us
     raise syntax 40.900 array ('"'fn'": CSV header (first line) does not start with a field named "OPTION" (written in any case)')

  do while csv~chars>0
     fields=csv~CSVLinein
     traceObj=.traceObject~new
     do counter c hdrField over arrHeaderFields
        value=fields[c]
        if hdrField="TIMESTAMP" then
           traceObj~setEntry(hdrField,.dateTime~fromIsoDate(value))
        else if value<>"" then
           traceObj~setEntry(hdrField,value)
     end
     toArray~append(traceObj)
  end
  csv~close
  return toArray




::routine quote               public
  use strict arg val, escape=.false
  if escape then
     return '"' || val~makeString~changeStr('"','""') || '"'
  return '"'val'"'


::routine writeFile
  use strict arg fn, data
  s=.stream~new(fn)
  s~~open("write replace") ~~charout(data) ~~close -- write all bytes
  return


/* ========================================================================= */




/* ========================================================================= */
/* Analyze trace objects and determine the guard state (hasGuardState; 'g'|'u')
 * at runtime and the wait state (isRunning; 'r'|'W')
 *
 * @param  collection
 * @return copy of recevied collection
*/
::routine annotateRuntimeState public
   use strict arg coll
   myArr=coll~makeArray

      -- sort by attributePools and nr (effective sequence of execution)
   myArr~sortWith(.sortByAttributePoolNr~new)

      -- define index names to be used (allows to change them easily if need arises)
   hasGuardStateIdx='HASGUARDSTATE'
   isRunningIdx    ='ISRUNNING'

   firstAttrPoolNr=-1
   do counter c traceObj over myArr
      if traceObj~attributePool=0 then
      do
         tmpTraceObject[hasGuardStateIdx]=.false   -- never in guarded state
         tmpTraceObject[isRunningIdx]    =.true    -- always runnable
         iterate
      end
      firstAttrPoolNr=c    -- save index of first traceObject with attribute pool
      leave
   end

   if firstAttrPoolNr=-1 then    -- nothing to do return sorted copy of received collection
      return myArr~sort
   diffAttrPools=1      -- calculate number of objects (different attribute pools)
   currAttrPoolNr=-1
   do i=firstAttrPoolNr to myArr~items
      tmpTraceObject=myArr[i]
         -- each new object's invocation starts out in the method's guard state and is running
      if tmpTraceObject~attributePool<>currAttrPoolNr then
      do
         diffAttrPools+=1
         currAttrPoolNr=tmpTraceObject~attributePool

         currHasGuardState=(tmpTraceObject~isGuarded=.true)
         tmpTraceObject[hasGuardStateIdx]=currHasGuardState
         currIsRunning=.true
         tmpTraceObject[isRunningIdx]    =currIsRunning
         iterate  -- now get next one
      end

      parse upper value tmpTraceObject["TRACELINE"] with . w0 w1 w2 w3
      -- if w0<>"*-*" then iterate  -- only annotate statement traceline

      -- currHasGuardState: can only be changed with "guard on" or "guard off"
      --                    for the same attribute pool, starts out with the
      --                    state of the method definition
      -- if in unguarded state: isRunning=.true
      -- if in guarded state: only if hasObjectLock=.true, then isRunning=.true

      if tmpTraceObject~hasObjectLock=.true then
      do
         hasObjectLock=(tmpTraceObject~hasObjectLock=.true)
         currHasGuardState=hasObjectLock
      end

      if currHasGuardState then  -- only running if holding object lock
      do
         tmpTraceObject[hasGuardStateIdx]=currHasGuardState
         currIsRunning=hasObjectLock
         tmpTraceObject[isRunningIdx]=currIsRunning
      end
      else  -- currently unguarded: always running
      do
         tmpTraceObject[hasGuardStateIdx]=currHasGuardState
         currIsRunning=.true
         tmpTraceObject[isRunningIdx]    =currIsRunning
         iterate
      end
      -- look for "guard on", "guard off", "guard on when ...", "guard off when..." in
      -- "*-* guard..." in TRACELINEs
      if w1="GUARD" then
         currHasGuardState=(w2="ON")   -- determine new state
   end
say "(debug)" .context~name": #" .line "- currAttrNr="currAttrPoolNr "(different objects)"
   return myArr         -- return annotated array




/* ========================================================================= */
/*
 * This comparator sorts the TraceObjects by interpreter, thread, invocation and nr.
*/
::class sortByThreadInvocationNr subclass Comparator  public -- sort by thread, invocation, nr
::method compare
   use arg left, right

   if left~interpreter > right~interpreter then return  1
   if left~interpreter < right~interpreter then return -1

   if left~thread > right~thread then return  1
   if left~thread < right~thread then return -1

   -- both are equal, now sort by invocation number
   if left~invocation > right~invocation then return  1
   if left~invocation < right~invocation then return -1

   -- bot are equal, now sort by nr which is the original trace sequence
   if left~nr > right~nr then return  1
   if left~nr < right~nr then return -1
   return 0    -- both are equal


/* ========================================================================= */
/*
 * This comparator sorts the TraceObjects by invocation and nr.
*/
::class sortByInvocationNr subclass Comparator public  -- sort by invocation, nr
::method compare
   use arg left, right

   if left~invocation > right~invocation then return  1
   if left~invocation < right~invocation then return -1

   -- bot are equal, now sort by nr which is the original trace sequence
   if left~nr > right~nr then return  1
   if left~nr < right~nr then return -1
   return 0    -- both are equal


/* ========================================================================= */
/*
 * This comparator is meant for easying the routine annotateRuntimeState, it
 * sorts the TraceObjects by attributePool and nr.
*/
::class sortByAttributePoolNr subclass Comparator  public -- sort by thread, invocation, nr
::method compare
   use arg left, right

   apLeft =left~attributePool
   apRight=right~attributePool

   if \apLeft~isNil, \apRight~isNil then
   do
      -- both attribute pool numbers are present
      if apLeft > apRight then return  1
      if apLeft < apRight then return -1
   end

   if apLeft~isNil, apRight~isNil then    -- sort by nr
   do
      -- bot are equal, now sort by nr which is the original trace sequence
      return sign(left~nr - right~nr)
   end

   if apLeft~isNil  then return -1
   if apRight~isNil then return 1
   return 0
