/* REXX ****************************************************************
* REXX EKA B show the border and modify points if necessary
* REXX EKA X creates a grid where one can estimate point coordinates
* REXX EKA Y creates grid + border
* REXX EKA W paint the area in white
* REXX EKA R replace the area
* Input:  EK.bmp
* Output: EKA*.bmp
* 11.06.2020 Walter Pachl
* 09.09.2020 Walter Pachl polish for the REXX Symposium
***********************************************************************/
Call time 'R'
dbg='dbg.txt'; 'erase' dbg
dbg=''
Parse Upper Arg which
fid='EK.bmp'
Select
  When which='B' Then suffix='B border'
  When which='X' Then suffix='X Grid'
  When which='Y' Then suffix='Y Grid+Border'
  When which='W' Then suffix='W White'
  When which='R' Then suffix='R Replacement'
  Otherwise Do
    Say 'Argument must be one of these: B X Y W R'
    Exit
    End
  End
tgt='EKA'suffix'.bmp'; 'erase "'tgt'"'

v.=0                                  /* indicates end of polygon     */
/*   line   x (measured in bytes)                                     */
v.1.1=  1 2300                        /* left border                  */
v.1.2=400 2600
v.1.3=410 2800
v.1.4=550 3000
v.1.5=700 3200
v.1.6=850 3900
v.1.7=900 4400

v.2.1=  1 5650                         /* right border                */
v.2.2=180 5720
v.2.3=240 5830
v.2.4=500 5700
v.2.5=800 5300
v.2.6=850 5200
v.2.7=900 4900

s=charin(fid,,40000000)                /* read bmp file               */
Say 'Picture has' length(s) 'bytes'

Parse Var s t +54 s                    /* Get header t and picture (s)*/

Parse Var t bfType +2,
            bfSize +4,
            bfReserved +4,
            bfOffBits +4,
            biSize +4,
            biWidth +4,
            biHeight +4

w=dnel(biWidth)                        /* Number of Pixels in a line  */
h=dnel(biHeight)                       /* Number of lines             */
Say 'Width ='w
Say 'Height='h

w3=3*w                                 /* Bytes in a line             */

ha=900                                 /* for test of the area, we    */
                                       /* don't need the replacement  */

Call poly 1                            /* left border                 */
Call poly 2                            /* right border                */

--d.=1
Do i=1 To 900                /* for each line in the replacement area */
  x1=x.1.i                             /* left border                 */
  x2=x.2.i                             /* right border                */
  d=x2-x1                              /* length of replacement       */
  x1=(x1%3)*3+1                        /* set to pixel boundary       */
  d=(d%3)*3                            /* make it multiple of 3       */
  x.i=x1                               /* start of replacement        */
  d.i=d                                /* length of replacement       */
  End

Do i=1 By 1 While s<>''                /* Now get the lines of the pic*/
  If i//500=0 Then say i               /* Show progress, now and then */
  Parse Var s l.i +(w3) s              /* Get line i                  */
  If i=ha & which<>'R' Then Leave      /* shorten it for test         */
  If which='R' Then Do         /* for Replace cut a piece of the line */
    If 450<i & i<=450+900 Then Do/* lines where we take replacement   */
      j=i-450                          /* line adjusted to 1...900    */
      r.j=substr(l.i,151,d.j)        /* replacement string for line j */
      End
    End
  End

blue ='111111110000000000000000'b
green='000000001111111100000000'b
red  ='000000000000000011111111'b
bg   ='111111111111111100000000'b  /* blue + green ->  turquoise  */
br   ='111111110000000011111111'b  /* blue + red   ->  lilac      */
gr   ='000000001111111111111111'b  /* green + red  ->  yellow     */

st.0=0
Do i=1 To 900              /* lines where replacements are to be made */
  /* This will show color bars in the left lower corner
  Select
    When i<100 Then l.i=overlay(copies(blue , 300),l.i,1)
    When i<200 Then l.i=overlay(copies(green, 200),l.i,1)
    When i<300 Then l.i=overlay(copies(red  , 100),l.i,1)
    When i<400 Then l.i=overlay(copies(bg   , 100),l.i,1)
    When i<500 Then l.i=overlay(copies(br   , 200),l.i,1)
    When i<600 Then l.i=overlay(copies(gr   , 300),l.i,1)
    Otherwise Nop
    End
  */
  Select
    When which='B' Then             /* B: show the border             */
      Call border
    When which='R' Then             /* R Replace with the cut pieces  */
      l.i=overlay(r.i,l.i,x.i)
    When which='X' Then             /* X Draw a grid                  */
      Call grid
    When which='Y' Then Do          /* Y Draw a grid and border       */
      Call grid
      Call border
      End
    Otherwise Do             /* Otherwise whiten the replacement area */
      ww=copies('11111111'b,d.i)
      l.i=overlay(ww,l.i,x.i)
      End
    End
  End
Do i=1 To h                             /* Neues BMP zusammenstellen */
  t=t||l.i
  End
Call charout tgt,t                       /* und schreiben             */
Say 'Created' tgt
Say time('E')
Exit

border:
  xa=x.i                                /* left border                */
  d=d.i                                 /* start of border line       */
  ww=copies('11111111'b,15)
  l.i=overlay(ww,l.i,xa)
  xb=x.i+d.i                            /* right border               */
  l.i=overlay(ww,l.i,xb)
  Return

grid:
  If i//200=0 Then Do                   /* some horizontal lines      */
    x1=2000
    i1=i+1
    i2=i+2
    ww=copies('11111111'b,4000)
    l.i=overlay(ww,l.i,x1)
    l.i1=overlay(ww,l.i1,x1)
    l.i2=overlay(ww,l.i2,x1)
    End
  ww=copies('11111111'b,15)
  Do x1=2400 To 6000 By 400             /* now some vertical lines    */
    x1=(x1%3)*3+1                       /* set to pixel boundary      */
    l.i=overlay(ww,l.i,x1)
    End
  Return

dnel: Procedure
/***********************************************************************
* compute the number from its representation (little endian)
***********************************************************************/
Parse Arg s
sr=reverse(s)
res=c2d(sr)
say 'dnel:' c2x(s) '=>' c2x(sr) '=>' c2d(sr)
Return  res

poly: Procedure Expose v. x. dbg
Parse Arg n                     /* n=1 left border, n=2 right border  */
/* compute i->x.n.i                                                   */
/* v.n.1, v.n.2, ...   points as (i x)                                */
Do pa=1 By 1                    /* index of a point                   */
  pb=pa+1                       /* index of next point                */
  If v.n.pb=0 Then              /* This is the last point             */
    Leave                       /* so we are done                     */
  Parse Var v.n.pa i1 x1        /* start of segment                   */
  Parse Var v.n.pb i2 x2        /* end of segment                     */
  dx=(x2-x1)/(i2-i1)            /* delta x from i to i+1              */
  Do i=i1 To i2                 /* walk through the segment           */
    x.n.i=x1+(i-i1)*dx          /* x coordinate in line i             */
    If n=1 & dbg<>'' Then
      Call lineout dbg,right(i,3) x.1.i
    End
  End
Return
